// renderer.js
const dom = {
    container: document.getElementById('widget-container'),
    rightColumn: document.getElementById('right-column'),
    timeline: document.getElementById('timeline'),
    statusText: document.getElementById('status-text'),
    artist: document.getElementById('artist'),
    titleWrapper: document.getElementById('title-wrapper'),
    titleSpans: document.querySelectorAll('.title-text'),
    pauseBtn: document.getElementById('pause-btn'),
    colorPicker: document.getElementById('color-picker'),
    dynamicFonts: document.getElementById('dynamic-fonts')
};

const state = {
    position: 0,
    duration: 0,
    scale: 1.0,
    currentTitle: '',
    // --- TIMER STATE ---
    syncPosition: 0,      // The last "true" position from the backend
    syncTime: 0,          // The system time we received the last update
    isTimerRunning: false // Flag to manage the interval
};

let timelineInterval;
let isCarouselEnabled = false;

function formatTime(seconds) { if (isNaN(seconds) || seconds < 0) return '00:00'; const mins = Math.floor(seconds / 60); const secs = Math.floor(seconds % 60); return `${String(mins).padStart(2, '0')}:${String(secs).padStart(2, '0')}`; }

// This function ONLY updates the DOM. It does no calculations.
function renderTimeline() {
    if (state.duration > 0) {
        dom.timeline.textContent = `${formatTime(state.position)} / ${formatTime(state.duration)}`;
    } else {
        dom.timeline.textContent = '--:-- / --:--';
    }
}

// --- START: FINAL TIMER LOGIC ---
function startTimer() {
    if (state.isTimerRunning) return; // Do nothing if it's already running
    state.isTimerRunning = true;

    // This interval is now the SOLE authority for updating the time display while playing.
    timelineInterval = setInterval(() => {
        const elapsedTime = (Date.now() - state.syncTime) / 1000;
        state.position = state.syncPosition + elapsedTime;
        renderTimeline();
    }, 250); // Update the display 4 times per second for a smooth look
}

function stopTimer() {
    clearInterval(timelineInterval);
    state.isTimerRunning = false;
}
// --- END: FINAL TIMER LOGIC ---

function updateCarousel() {
    const isOverflowing = dom.titleSpans[0].scrollWidth > dom.rightColumn.clientWidth;
    if (isCarouselEnabled && isOverflowing) {
        const duration = dom.titleSpans[0].textContent.length / 8;
        dom.titleWrapper.style.animationDuration = `${Math.max(5, duration)}s`;
        dom.titleWrapper.classList.add('is-overflowing');
    } else {
        dom.titleWrapper.classList.remove('is-overflowing');
        dom.titleWrapper.style.animationDuration = '0s';
    }
}

// --- START: FINAL MEDIA HANDLER ---
window.api.on('update-media', (mediaInfo) => {
    const isPlaying = mediaInfo.status === 'Playing';
    const newTitle = mediaInfo.title || 'No media playing';

    if (newTitle !== state.currentTitle) {
        state.currentTitle = newTitle;
        dom.titleSpans.forEach(span => span.textContent = newTitle);
        setTimeout(updateCarousel, 0);
    }

    if (mediaInfo.title) {
        dom.artist.textContent = mediaInfo.artist;
        dom.statusText.textContent = isPlaying ? 'NOW PLAYING' : 'PAUSED';
        dom.pauseBtn.textContent = isPlaying ? 'pause' : 'play';
    } else {
        dom.artist.textContent = 'Desktop';
        dom.statusText.textContent = 'IDLE';
    }

    // Handle timeline info
    if (mediaInfo.timeline && mediaInfo.timeline.duration > 0) {
        state.duration = mediaInfo.timeline.duration;

        // **THE CORE FIX**: We ONLY update the "anchor" data.
        // We never call renderTimeline() from here when playing.
        state.syncPosition = mediaInfo.timeline.position;
        state.syncTime = Date.now();

        if (isPlaying) {
            // Ensure the timer is running. It will pick up the new sync data on its own.
            startTimer();
        } else {
            // If paused, stop the local timer and set the time manually.
            stopTimer();
            state.position = state.syncPosition;
            renderTimeline(); // Render the final, paused time.
        }
    } else {
        // No valid timeline info, stop the timer and reset the display.
        stopTimer();
        state.duration = 0;
        state.position = 0;
        renderTimeline();
    }
});
// --- END: FINAL MEDIA HANDLER ---

window.api.on('apply-settings', (settings) => {
    state.scale = settings.scale;
    document.documentElement.style.setProperty('--ui-scale', settings.scale);
    document.documentElement.style.setProperty('--ui-opacity', settings.opacity);
    document.documentElement.style.setProperty('--main-color', settings.color);
    document.documentElement.style.setProperty('--main-font', `'${settings.font}', sans-serif`);
    isCarouselEnabled = settings.carousel;
    setTimeout(updateCarousel, 0);
});

const resizeObserver = new ResizeObserver(entries => {
    for (let entry of entries) {
        const scaledBounds = entry.target.getBoundingClientRect();
        window.api.send('request-resize', { width: scaledBounds.width, height: scaledBounds.height });
    }
});
resizeObserver.observe(dom.container);

window.api.on('set-scale', (scale) => {
    state.scale = scale;
    document.documentElement.style.setProperty('--ui-scale', scale);
});

dom.titleWrapper.addEventListener('mouseenter', () => {
    if (dom.titleWrapper.classList.contains('is-overflowing')) {
        dom.titleWrapper.style.animationPlayState = 'running';
    }
});
dom.titleWrapper.addEventListener('mouseleave', () => {
    dom.titleWrapper.style.animationPlayState = 'paused';
});

let isMKeyPressed = false;
dom.container.addEventListener('mouseenter', () => window.api.send('set-interactive'));
dom.container.addEventListener('mouseleave', () => { isMKeyPressed = false; window.api.send('set-click-through'); });
window.addEventListener('keydown', (e) => { if (e.key.toLowerCase() === 'm') isMKeyPressed = true; });
window.addEventListener('keyup', (e) => { if (e.key.toLowerCase() === 'm') isMKeyPressed = false; });
dom.container.addEventListener('mousedown', (e) => {
    if (isMKeyPressed && e.button === 0) {
        let initialMouse = { x: e.screenX, y: e.screenY };
        const onMouseMove = (moveEvent) => {
            const delta = { deltaX: moveEvent.screenX - initialMouse.x, deltaY: moveEvent.screenY - initialMouse.y };
            initialMouse = { x: moveEvent.screenX, y: moveEvent.screenY };
            window.api.send('move-window', delta);
        };
        const onMouseUp = () => { document.removeEventListener('mousemove', onMouseMove); document.removeEventListener('mouseup', onMouseUp); };
        document.addEventListener('mousemove', onMouseMove);
        document.addEventListener('mouseup', onMouseUp);
    }
});
dom.container.addEventListener('contextmenu', (e) => { if (!isMKeyPressed) e.preventDefault(); });

document.getElementById('prev-btn').addEventListener('click', () => window.api.send('media-control', 'previous'));
document.getElementById('pause-btn').addEventListener('click', () => window.api.send('media-control', 'play_pause'));
document.getElementById('next-btn').addEventListener('click', () => window.api.send('media-control', 'next'));
window.api.on('toggle-carousel', (isEnabled) => { isCarouselEnabled = isEnabled; setTimeout(updateCarousel, 0); });
window.api.on('set-opacity', (value) => document.documentElement.style.setProperty('--ui-opacity', value));
window.api.on('open-color-picker', () => dom.colorPicker.click());
dom.colorPicker.addEventListener('input', (event) => { const newColor = event.target.value; document.documentElement.style.setProperty('--main-color', newColor); window.api.send('save-setting', { key: 'color', value: newColor }); });
window.api.on('set-font', (fontName) => { document.documentElement.style.setProperty('--main-font', `'${fontName}', sans-serif`); });

async function loadFonts() { try { const fontFiles = await window.api.invoke('get-fonts'); let fontFaceRules = fontFiles.map(file => { const fontName = file.split('.').slice(0, -1).join('.'); return `@font-face { font-family: '${fontName}'; src: url('fonts/${file}'); }`; }).join('\n'); dom.dynamicFonts.innerHTML = fontFaceRules; } catch (error) { console.error("Failed to load custom fonts:", error); } }
loadFonts();