const { app, BrowserWindow, ipcMain, Menu } = require('electron');
const path = require('path');
const fs = require('fs');
const Store = require('./store.js');
const fetch = (...args) => import('node-fetch').then(({default: fetch}) => fetch(...args));

const BASE_WIDTH = 550;
const BASE_HEIGHT = 180;

const store = new Store({
    configName: 'user-preferences-v5-html-input',
    defaults: {
        opacity: 1.0,
        fontFamily: 'Poppins',
        color: '#FFFFFF',
        scale: 1.0,
        bounds: { x: 50, y: 50, width: BASE_WIDTH, height: BASE_HEIGHT },
        weather: {
            locationName: 'New York',
            units: 'C'
        }
    }
});

async function fetchWeather(win, location, units) {
    if (!location) {
        win.webContents.send('update-weather', { error: 'Location not set.' });
        return;
    }
    const url = `https://v2.wttr.in/${encodeURIComponent(location)}?format=j1`;
    try {
        const response = await fetch(url);
        const data = await response.json();
        if (!data.current_condition || data.current_condition.length === 0) {
            win.webContents.send('update-weather', { error: 'Location not found.' });
            return;
        }
        const current = data.current_condition[0];
        const temp = units === 'C' ? current.temp_C : current.temp_F;
        const weatherInfo = {
            temp: Math.round(temp),
            weatherCode: current.weatherCode,
            description: current.weatherDesc[0].value,
            unit: `°${units}`,
            locationName: location
        };
        win.webContents.send('update-weather', weatherInfo);
    } catch (error) {
        win.webContents.send('update-weather', { error: 'Network error.' });
    }
}

function createWindow() {
    const savedBounds = store.get().bounds;
    const win = new BrowserWindow({
        ...savedBounds,
        frame: false, transparent: true, skipTaskbar: true, resizable: false, type: 'desktop',
        webPreferences: { preload: path.join(__dirname, 'preload.js'), nodeIntegration: false, contextIsolation: true }
    });

    win.setIgnoreMouseEvents(true, { forward: true });
    win.on('moved', () => { store.set('bounds', win.getBounds()); });
    win.loadFile('index.html');

    win.webContents.on('did-finish-load', () => {
        win.webContents.send('load-settings', store.get());
        const initialSettings = store.get().weather;
        fetchWeather(win, initialSettings.locationName, initialSettings.units);
    });

    setInterval(() => {
        const currentSettings = store.get().weather;
        fetchWeather(win, currentSettings.locationName, currentSettings.units);
    }, 30 * 60 * 1000);

    ipcMain.on('mouse-enter-widget', () => { win.setIgnoreMouseEvents(false); });
    ipcMain.on('mouse-leave-widget', () => { win.setIgnoreMouseEvents(true, { forward: true }); });

    let initialDragState = { mouse: { x: 0, y: 0 }, window: { x: 0, y: 0 } };
    ipcMain.on('start-drag', (event, { mouseX, mouseY }) => {
        const windowBounds = win.getBounds();
        initialDragState = { mouse: { x: mouseX, y: mouseY }, window: { x: windowBounds.x, y: windowBounds.y } };
    });
    ipcMain.on('drag', (event, { mouseX, mouseY }) => {
        const deltaX = mouseX - initialDragState.mouse.x;
        const deltaY = mouseY - initialDragState.mouse.y;
        win.setPosition(initialDragState.window.x + deltaX, initialDragState.window.y + deltaY);
    });
    ipcMain.on('end-drag', () => { store.set('bounds', win.getBounds()); });

    ipcMain.on('set-new-location', (event, locationName) => {
        if (locationName) {
            const currentSettings = store.get();
            store.set('weather', { ...currentSettings.weather, locationName: locationName });
            fetchWeather(win, locationName, currentSettings.weather.units);
        }
    });

    ipcMain.on('show-context-menu', (event) => {
        const currentSettings = store.get();
        
        const appearanceSubmenu = [
            { label: 'Text Size', submenu: [50, 75, 100, 125, 150, 200, 250, 300].map(s => {
                const scaleValue = s / 100.0;
                return {
                    label: `${s}%`, type: 'radio', checked: currentSettings.scale === scaleValue,
                    click: () => {
                        const newWidth = Math.round(BASE_WIDTH * scaleValue);
                        const newHeight = Math.round(BASE_HEIGHT * scaleValue);
                        win.setSize(newWidth, newHeight, true);
                        const currentBounds = win.getBounds();
                        store.set('scale', scaleValue);
                        store.set('bounds', {
                            x: currentBounds.x,
                            y: currentBounds.y,
                            width: newWidth,
                            height: newHeight
                        });
                        win.webContents.send('load-settings', store.get());
                    }
                }
            })},
            { label: 'Font Style', submenu: [
                // --- CORRECTED FONT LIST (NO SEPARATORS) ---
                { name: 'Poppins (Modern)', value: 'Poppins' },
                { name: 'Roboto (Clean)', value: 'Roboto' },
                { name: 'Montserrat (Stylish)', value: 'Montserrat' },
                { name: 'Lato (Elegant)', value: 'Lato' },
                { name: 'Oswald (Bold)', value: 'Oswald' },
                { name: 'Nunito (Rounded)', value: 'Nunito' },
                { name: 'Segoe UI (Windows)', value: 'Segoe UI' },
                { name: 'Calibri (Office)', value: 'Calibri' },
                { name: 'Arial (Classic)', value: 'Arial' },
                { name: 'Helvetica (Professional)', value: 'Helvetica' },
                { name: 'Verdana (Web Safe)', value: 'Verdana' },
                { name: 'Tahoma (Compact)', value: 'Tahoma' },
                { name: 'Georgia (Serif)', value: 'Georgia' },
                { name: 'Times New Roman', value: 'Times New Roman' },
                { name: 'Garamond (Formal)', value: 'Garamond' },
                { name: 'Impact (Headline)', value: 'Impact' },
                { name: 'Arial Black', value: 'Arial Black' },
                { name: 'Consolas (Code)', value: 'Consolas' },
                { name: 'Courier New', value: '"Courier New", monospace' },
                { name: 'Lucida Console', value: 'Lucida Console' }
            ].map(font => ({
                label: font.name, type: 'radio', checked: currentSettings.fontFamily === font.value,
                click: () => { store.set('fontFamily', font.value); win.webContents.send('load-settings', store.get()); }
            }))},
            { label: 'Font Color', submenu: [
                // --- CORRECTED COLOR PALETTE (NO SEPARATORS) ---
                { name: 'Classic White', value: '#FFFFFF' },
                { name: 'Sleek Black', value: '#000000' },
                { name: 'Light Grey', value: '#CCCCCC' },
                { name: 'Charcoal Grey', value: '#36454F' },
                { name: 'Cyber Yellow', value: '#FFD300' },
                { name: 'Gold', value: '#FFD700' },
                { name: 'Vivid Orange', value: '#FF5F00' },
                { name: 'Tomato Red', value: '#FF6347' },
                { name: 'Crimson Red', value: '#DC143C' },
                { name: 'Electric Blue', value: '#7DF9FF' },
                { name: 'Deep Sky Blue', value: '#00BFFF' },
                { name: 'Royal Blue', value: '#4169E1' },
                { name: 'Lime Green', value: '#32CD32' },
                { name: 'Forest Green', value: '#228B22' },
                { name: 'Teal', value: '#008080' },
                { name: 'Hot Pink', value: '#FF1D8E' },
                { name: 'Orchid', value: '#DA70D6' },
                { name: 'Lavender', value: '#E6E6FA' },
                { name: 'Deep Purple', value: '#9400D3' }
            ].map(color => ({
                label: color.name, type: 'radio', checked: currentSettings.color === color.value,
                click: () => { store.set('color', color.value); win.webContents.send('load-settings', store.get()); }
            }))}
        ];

        const weatherSettingsSubmenu = [
            { 
                label: 'Set Location', 
                click: () => {
                    win.webContents.send('show-location-input');
                }
            },
            { type: 'separator' },
            { label: 'Units', submenu: [
                { label: 'Celsius (°C)', type: 'radio', checked: currentSettings.weather.units === 'C', click: () => {
                    const newSettings = { ...currentSettings.weather, units: 'C' };
                    store.set('weather', newSettings);
                    fetchWeather(win, newSettings.locationName, newSettings.units);
                }},
                { label: 'Fahrenheit (°F)', type: 'radio', checked: currentSettings.weather.units === 'F', click: () => {
                    const newSettings = { ...currentSettings.weather, units: 'F' };
                    store.set('weather', newSettings);
                    fetchWeather(win, newSettings.locationName, newSettings.units);
                }}
            ]}
        ];

        const template = [
            { label: 'Appearance', submenu: appearanceSubmenu },
            { label: 'Weather Settings', submenu: weatherSettingsSubmenu },
            { label: 'Transparency', submenu: [100, 90, 80, 70, 60, 50, 40, 30, 20, 10].map(p => ({
                label: `${p}%`, type: 'radio', checked: currentSettings.opacity === (p / 100.0),
                click: () => { store.set('opacity', p / 100.0); win.webContents.send('load-settings', store.get()); }
            }))},
            { type: 'separator' },
            {
                label: 'Reset Widget',
                click: () => {
                    try {
                        if (fs.existsSync(store.path)) {
                            fs.unlinkSync(store.path);
                        }
                        app.relaunch();
                        app.quit();
                    } catch (e) {
                        console.error('Failed to reset widget:', e);
                    }
                }
            },
            { label: 'Close Widget', role: 'quit' }
        ];
        const menu = Menu.buildFromTemplate(template);
        menu.popup({ window: win });
    });
}

app.whenReady().then(createWindow);
app.on('window-all-closed', () => { if (process.platform !== 'darwin') app.quit(); });